/* 
 * ParentSelection.h - Parent selection using X-tournament selection
 *
 * Copyright (C) 1998 J.I. van Hemert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: J.I. van Hemert <jvhemert@cs.leidenuniv.nl>
 *
 * Last Revision: 10-Mar-1999 21:56 by jvhemert
 *
 */

#ifndef ParentSelection_H
#define ParentSelection_H

#include <vector>
#include <iostream>

#include "Population.h"
#include "Individual.h"
#include "Random.h"
#include "Error.h"


template <class F, class D>
class ParentSelection
{
	public:

		ParentSelection
		(
			unsigned int _selectionsize,
			Random<unsigned int> const & _random,
			unsigned int _tournamentsize
		)
		: 
			selection(0),
			selectionsize(_selectionsize),
			random(_random),
			tournamentsize(_tournamentsize)
		{ 
		}

		~ParentSelection(){};

		unsigned int GetSize() const
		{
			 return selectionsize;
		};

		void
		operator()
		(
			Population<F, D> const & _population
		);

		friend class const_iterator
		{
			public:

				const_iterator
				(
					ParentSelection const & _parentselection
				)
				:
					parentselection(_parentselection)
				{
					First();
				};

				~const_iterator(){};

				Individual<F, D> const &
				operator* () const
				{
					if (current == parentselection.selection.end())
					{
						 throw Error("tried to read past end of parentselection in parentselection::operator*()");
					}
					return *(*current);
				};

				void
				First()
				{
					current = parentselection.selection.begin();
				};
				
				Individual<F, D> const &
				operator++(int)
				{
					if (current == parentselection.selection.end())
					{
						 throw Error("tried to read past end of parentselection in parentselection::operator++(int)");
					}

					return **(current++);
				};

				Individual<F, D> const &
				operator++()
				{
					if (current == parentselection.selection.end())
					{
						 throw Error("tried to read past end of parentselection in parentselection::operator++()");
					}

					return **(++current);
				};
				
				bool
				End() const
				{
					return (current == parentselection.selection.end());
				};

			private:

				ParentSelection const & parentselection;

				vector<Individual<F, D> const *>::const_iterator current;

		};
		
		friend ostream & operator<<
		(
			ostream & _ostream,
			ParentSelection const & _parentselection
		)
		{
			for (vector<Individual<F, D> const *>::const_iterator pi = _parentselection.selection.begin(); pi != _parentselection.selection.end(); ++pi)
			{
				_ostream << *(*pi) << endl;
			}

			return _ostream;
		};

	protected:

	private:

		vector<Individual<F, D> const *> selection;

		unsigned int selectionsize;

		Random<unsigned int> const & random;

		unsigned int tournamentsize;

}; // ParentSelection


template <class F, class D>
void
ParentSelection<F, D>::operator()
(
	Population<F, D> const & _population
)
{
	selection.erase(selection.begin(), selection.end());

	for (unsigned int i = 0; i < selectionsize; ++i)
	{
		unsigned int bestindex = random(0, _population.GetSize());

		for (unsigned int j = 0; j < tournamentsize - 1; ++j)
		{
			unsigned int index = random(0, _population.GetSize());
			if (_population.GetConstIndividual(bestindex) < _population.GetConstIndividual(index))
			{
				 bestindex = index;
			}
		}
		
		selection.push_back(&_population.GetConstIndividual(bestindex));
	}

} // operator()


/*
template <class F, class D>
void
ParentSelection<F, D>::operator()
(
	Population const & _population
)
{
	double LinearRankingBias = 1.5;
	double PreWork1 = LinearRankingBias * LinearRankingBias;
	double PreWork2 = 4.0 * (LinearRankingBias - 1);
	double PreWork3 = 2.0 * (LinearRankingBias - 1);

	selection.erase(selection.begin(), selection.end());

	for (unsigned int i = 0; i < selectionsize; ++i)
	{
		unsigned int index = (unsigned int) ((double) _population.GetSize() * (LinearRankingBias - sqrt (PreWork1 - PreWork2 * random(0,1))) / PreWork3);
		
		selection.push_back(&_population.GetConstIndividual(index));
	}

} // operator()
*/

#endif // ParentSelection_H

// Eof ParentSelection.h
