/* 
 * EA.cc - An evolutionary algorithm with one population.
 *
 * Copyright (C) 1998 J.I. van Hemert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: J.I. van Hemert <jvhemert@cs.leidenuniv.nl>
 *
 * Last Revision: 10-Mar-1999 21:54 by jvhemert
 *
 */

#ifndef EA_H
#define EA_H

#include "Population.h"
#include "Random.h"
#include "Reproduction.h"
#include "SurvivorsSelection.h"
#include "Configuration.h"

template <class F, class D>
class EA
{
	public:

		EA
		(
		 	Configuration const & _configuration,
			Initialization<D> const & _initialization,
			Evaluate<F, D> const & _evaluate
		)
		: 
			configuration(_configuration),
			population(_initialization, _evaluate, configuration.AsUnsignedInt("PopulationSize")),
			numberofgenerations(1)
		{
		}

		~EA(){};

		void GotoNextGeneration
		(
			ParentSelection<F, D> & _parentselection,
			Reproduction<F, D> & _reproduction,
			SurvivorsSelection<F, D> & _survivorsselect
	 	);

		unsigned long int
		GetNumberOfGenerations()
		{
			 return NumberOfGenerations;
		};

		Population<F, D> &
		GetPopulation() 
		{
			return population;	 
		};
		
		Population<F, D> const &
		GetConstPopulation() const
		{
			return population;	 
		};
		
	protected:

	private:

		Configuration const & configuration;

		Population<F, D> population;

		unsigned long int numberofgenerations;

}; // EA

template <class F, class D>
void
EA<F, D>::GotoNextGeneration
(
	ParentSelection<F, D> & _parentselection,
	Reproduction<F, D> & _reproduction,
	SurvivorsSelection<F, D> & _survivorsselect
)
{
	// Make a selection of parents
	_parentselection(population);

#ifdef DEBUG
	cerr << "[EA::GotoNextGeneration (Parent Selection)]" << endl << _parentselection << endl;
#endif

	// Produce offspring using the reproduction mechanism and the selection of parents
	list<Individual<F, D> > offspring = _reproduction(_parentselection);


#ifdef DEBUG
	cerr << "[EA::GotoNextGeneration (Offspring)]" << endl;
	copy(offspring.begin(), offspring.end(), ostream_iterator<Individual<F, D> >(cerr, "\n"));
#endif

	// Merge offspring and population to form the new generation
	_survivorsselect(population, offspring);

#ifdef DEBUG
	cerr << "[EA::GotoNextGeneration (New population)]" << endl << population << endl;
#endif

	numberofgenerations++;

} // GotoNextGeneration


#endif // EA_H

// Eof EA.h
