/* 
 * Data.h - Storage for a Mondriaan painting
 *
 * Copyright (C) 1998 J.I. van Hemert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: J.I. van Hemert <jvhemert@cs.leidenuniv.nl>
 *
 * Last Revision: 13-Mar-1999 17:24 by jvhemert
 *
 */

#ifndef Data_H
#define Data_H

#include "Canvas.h"
#include <vector>

class Data
{
	public:

		Data
		()
		:
			contents(0),
			colors(0)
		{};
		
		Data
		(
			unsigned int _size
		)
		:
			contents(_size),
			colors(_size)
		{};
		
		~Data(){ };

		vector<unsigned int> contents;
		vector<unsigned int> colors;


		// Draw a Mondriaan-like image on a canvas.

			void
		Draw
		(
			Canvas & _canvas
		)
		{
			// First draw a grid.
			for (unsigned int i = 0; i < 15; ++i)
			{
				_canvas.DrawVerticalLine(Coordinates(i, 0), Coordinates(i, 16), Color::black);
				_canvas.DrawHorizontalLine(Coordinates(0, i), Coordinates(16, i), Color::black);
				 
			}

			// Finally, draw rectangles on the grid-filled canvas.
			unsigned int i = 0;
			Split(i, _canvas, Coordinates(0, 0), Coordinates(15, 15));

		} // Draw;

		friend
		ostream &
		operator<<
		(
			ostream & _ostream,
			Data const & _data
		)
		{
			_ostream << "Contents: ";
			for (vector<unsigned int>::const_iterator i = _data.contents.begin(); i != _data.contents.end(); ++i)
			{
				//_ostream << (*i ? '1' : '0');
				_ostream << (*i) << " ";
			}

			_ostream << "Colors: ";
			for (vector<unsigned int>::const_iterator i = _data.colors.begin(); i != _data.colors.end(); ++i)
			{
				_ostream << (*i) << " ";
			}


			return _ostream;
		};

	protected:

	private:

		/* Used by Split to draw rectangles on the canvas, using the given index
		 * number to choose a color.
		 */

			void
		DrawRectangleWithSomeColor
		(
			Canvas & _canvas,
			Coordinates const & _corner1,
			Coordinates const & _corner2,
			unsigned int i
		)
		{
			switch (colors[i])
			{
				case 0:
					_canvas.DrawRectangle(_corner1, _corner2, Color::white);
					break;
				case 1:
					_canvas.DrawRectangle(_corner1, _corner2, Color::red);
					break;
				case 2:
					_canvas.DrawRectangle(_corner1, _corner2, Color::blue);
					break;
				case 3:
					_canvas.DrawRectangle(_corner1, _corner2, Color::yellow);
					break;
				case 4:
					_canvas.DrawRectangle(_corner1, _corner2, Color::black);
					break;
				case 5:
					_canvas.DrawRectangle(_corner1, _corner2, Color::Color(150,150,150));
					break;
				default:
					throw Error("The domain for colors is set too high (> 6).");
			}
		};
			

	
		/* Recursively split up the large rectangle (canvas as a whole) in
		 * smaller sections using double recursion (horizontal/vertical). The
		 * genotype is used to determine wheter to split and how to split, also
		 * every second number is used to determine the color of a rectangle.
		 * This is the decoder part of the data.
		 */

			void
		Split
		(
		 	unsigned int & i,
		 	Canvas & _canvas,
			Coordinates _corner1,
			Coordinates _corner2
		)
		{
			if (contents[i] > 2)
			{
				throw Error("The rectangles domain is set too high (> 3).");	 
			}
			
			if (contents[i] == 0)
			{
				DrawRectangleWithSomeColor(_canvas, _corner1, _corner2, i);
			}
			else
			{
				if ((_corner1.y + 1 < _corner2.y) && (contents[i] == 1))
				{
					Coordinates corner1 = _corner1;
					Coordinates corner2 = _corner2;

					_corner1.y = (unsigned int) ((_corner1.y + _corner2.y) * 0.5);
					_corner2.y = _corner1.y - 1;
					i++;
					Split(i, _canvas, corner1, _corner2);
					i++;
					Split(i, _canvas, _corner1, corner2);
				}
				else
				{
					if ((_corner1.x + 1 < _corner2.x) && (contents[i] == 2))
					{
						Coordinates corner1 = _corner1;
						Coordinates corner2 = _corner2;

						_corner1.x = (unsigned int) ((_corner1.x + _corner2.x) * 0.5);
						_corner2.x = _corner1.x - 1;
						i++;
						Split(i, _canvas, corner1, _corner2);
						i++;
						Split(i, _canvas, _corner1, corner2);
					}
					else
					{
						DrawRectangleWithSomeColor(_canvas,  _corner1, _corner2, i);
					}
				}

			}

			
		} // Split;


			
};

#endif // Data_H

// Eof Data.h
