/* 
 * Canvas.cc - A canvas to draw upon. It handles colored squares, vertical lines
 * and horizontal lines.
 *
 * Copyright (C) 1998 J.I. van Hemert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: J.I. van Hemert <jvhemert@cs.leidenuniv.nl>
 *
 * Last Revision: 10-Mar-1999 19:28 by jvhemert
 *
 */

#ifndef Canvas_H
#define Canvas_H

#include <gtk/gtk.h>
#include <iostream>

//
// This class is a simple container for holding a set of coordinates in two dimensions.

class Coordinates
{
	public:

		Coordinates
		(
			unsigned long int _x,
			unsigned long int _y
		)
		:
			x(_x),
			y(_y)
		{
		};

		Coordinates()
		:
			x(0),
			y(0)
		{};

		unsigned long int x;
		unsigned long int y;

		friend 
			ostream &
		operator<<
		(
			ostream & _ostream,
			Coordinates const & _coordinates
		)
		{
			 _ostream << "(" << _coordinates.x << ", " << _coordinates.y << ")";

			 return _ostream;
		};

		// For comparing two sets of coordinates

			Coordinates const &
		operator=
		(
			Coordinates const & _coordinates
		)
		{
			x = _coordinates.x;
			y = _coordinates.y;

			return *this;
		};

			bool
		operator==
		(
			Coordinates const & _coordinates
		)
		{
			 return ( (_coordinates.x == x) && (_coordinates.y ==y ));
		};

			bool
		operator!=
		(
			Coordinates const & _coordinates
		)
		{
			 return ( (_coordinates.x != x) || (_coordinates.y !=y ));
		};
};


// This class holds a color coded as a set of three numbers between 0 and 255. These
// are the values for red, green and blue. A list of some simple colors is provided.

class Color
{
	public:

		Color
		(
			short int _r = 0,
			short int _g = 0,
			short int _b = 0
		)
		:
			r(_r),
			g(_g),
			b(_b)
		{};

		short int r;
		short int g;
		short int b;

		static Color black;
		static Color white;
		static Color red;
		static Color green;
		static Color blue;
		static Color yellow;
		static Color magenta;
		static Color cyan;

		friend 
			ostream &
		operator<<
		(
			ostream & _ostream,
			Color const & _color
		)
		{
			 _ostream << "(" << _color.r << ", " << _color.g << ", " << _color.b << ")";

			 return _ostream;
		};

	private:

};


// The canvas requires to be linked to a pixmap (Gtk) to draw upon, it also requires
// the size of the pixmap in pixels. Drawing on the canvas is done using three functions
// (DrawRectangle, DrawVerticalLine, DrawHorizontalLine) which makes use of a simpler
// smaller coordinate system. Here set between (0,0) and (15,15).

class Canvas
{
	public:


		Canvas
		(
			GdkPixmap *_pixmap,
			Coordinates _size
		);

		~Canvas();

			void
		Clear();

			void 
		DrawRectangle
		(
			Coordinates const & _corner1,
			Coordinates const & _corner2,
			Color const & _color
		);

			void 
		DrawVerticalLine
		(
			Coordinates const & _corner1,
			Coordinates const & _corner2,
			Color const & _color
		);

			void 
		DrawHorizontalLine
		(
			Coordinates const & _corner1,
			Coordinates const & _corner2,
			Color const & _color
		);

	protected:

	private:

		GdkPixmap *pixmap;

		Coordinates size;

		static unsigned int scale;

}; // Canvas

#endif // Canvas_H

// Eof Canvas.h
